//================================================================
//
// cutSequenceObject.js   
//
// Object which displays and manages sequencing control when cutting
//
//

// ----------------------------------------------------------------------------
// COPYRIGHT 2003 DELCAM PLC., BIRMINGHAM, ENGLAND. 
// ----------------------------------------------------------------------------
//
// History.
// Who When       What   
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

//=== cutSequence Constructor ==============================================
//
// Set up function pointers and properties
//
// History
// Who When       What
// --- -------- ---------------------------------------------------------
// bem 26/09/03 Written
// bem 26/11/03 Added support for optional control of locking start points
// bem 25/02/05 Add &nbsp; padding to stop japanese text breaking over multiple lines
//-----------------------------------------------------------------------------

function CutSequence
   (
   name,
   show_lock_start_points,
   changed_callback 
   )
{
   // Assign Page Members
   this.name = name;

   this.autoSequence = true;       // we never default to manual sequencing
   this.autoSequenceType = 0;
   this.lockStartPoints = true;

   if (show_lock_start_points == null)
      this.showLockStartPoints = false;
   else
      this.showLockStartPoints = show_lock_start_points;

   this.selectingVectors = false;  // flag indicating if we are in vector selection mode
   // members used to store vector used to manually control sequencing and original vectors
   this.ToolpathOriginalVectorSelection = "";
   this.SequenceDriveVectorId = "";


   // allow setting of default sequence option from registry
   this.sequenceType = artcam.RetrieveInt("SequenceSettings","DefaultSequenceType",0);
    // Assign Functions
   this.writeDiv		             = cutSequence_WriteDiv;
   this.initialise		          = cutSequence_Initialise;

   this.onToggleTerse             = cutSequence_onCutDirectionTerseClick;
   this.onSequenceTypeRadio       = cutSequence_onSequenceTypeRadio;
   this.onSequenceSelect          = cutSequence_onSequenceSelect;
   this.onVectorSelectionChanged  = cutSequence_VectorSelectionChanged;
   this.onCancelVectorSelection   = cutSequence_onCancelVectorSelection;
   this.onSequenceSelectVector    = cutSequence_onSequenceSelectVector;
   this.onSequenceLockStartPoints = cutSequence_onSequenceLockStartPoints;

   this.updateParametersFromToolpath = cutSequence_UpdateParametersFromToolpath;
   this.saveParametersToToolpath     = cutSequence_SaveParametersToToolpath;

   this.getManualSequenceVectorId    = cutSequence_getManualSequenceVectorId;
   this.setAutoSequenceType          = cutSequence_setAutoSequenceType;
   this.getAutoSequenceType          = cutSequence_getAutoSequenceType;
   this.ValidateVector               = cutSequence_ValidateVector;
   this.SetSequenceVectorArrows      = cutSequence_SetSequenceVectorArrows;
   this.PopulateSequenceList         = cutSequence_PopulateSequenceList;
   this.CleanUp                      = cutSequence_CleanUp;

   // is there a callback ?
   if (changed_callback == null)
      {
      this.doCallback = false;
      }
   else
      {
      this.updateCallback = changed_callback
      this.doCallback = true;
      }

   // get path to the root of our html - we use this to build
   // addresses to our images
   this.image_dir = artcam.HtmlRootDir();
   this.image_dir += "..\\SharedImages\\";
}

// === cutSequence_Initialise ========================================
//
// Initialise dropdowns etc for this object
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_Initialise()
{

   // Set up members for the data controls
   this.objSequenceRow             = eval(this.name + "jSequenceRow"); // show / hide this to hide all data
   this.objSequenceTypeList        = eval(this.name + "jSequenceTypeList");
   this.objSequenceAutoRadio       = eval(this.name + "jSequenceAutoRadio");
   if (this.showLockStartPoints)
      this.objSequenceLockStartPoints = eval(this.name + "jSequenceLockStartPointsCheck");

   this.objSequenceSelectVectorButton = eval(this.name + "jSequenceSelectVectorButton");
   this.objSequenceVectorSelectedSpan = eval(this.name + "jSequenceVectorSelectedSpan");

   this.OpenCloseButton  = eval( this.name + "OpenCloseButton");

   PromoteDivToSelect(this.objSequenceTypeList,this.image_dir);

   this.PopulateSequenceList();

   // select automatic sequencing by default
   this.onSequenceTypeRadio(0,false);
}


//=== cutSequence_WriteDiv ==============================================
//
// Writes out the html which is the interface for the cut sequence
// control into any toolpath dialog
//
// History
// Who When       What
// --- -------- ---------------------------------------------------------
// bem 26/09/03 Written
// dmr 11/09/07 Changes for licensing options
//-----------------------------------------------------------------------

function cutSequence_WriteDiv()
{

   // This is a bit like create. We write all our document objects into the page
   // then show and hide them as we wish.
   document.writeln("<tr featureType='FT_2DMACH_SEQUENCING'>");
   document.writeln("<td valign=top><img src='" + this.image_dir + "/order.gif' align=top></td>");  
   document.writeln("<td>");

   document.writeln("<table width=100% border=0 cellspacing=0>");
   document.writeln("  <tr><td colspan=2 height=2px></td></tr>");
   document.writeln("  <tr><td colspan=2 bgcolor=buttonshadow></td></tr>");
   document.writeln("  <tr><td colspan=2 bgcolor=buttonhighlight></td></tr>");
   document.writeln("  <tr><td colspan=2 height=2px></td></tr>");
   document.writeln("    <tr id='"+this.name+"jSequenceSummary' bgcolor=" + artcam.GetSystemColourShadeString(COLOR_BTNFACE,2,COLOR_BTNSHADOW,1) + ">");
   document.writeln("      <td colspan=2 onclick='" + this.name + ".onToggleTerse()'>");
   document.writeln("          <img id='" + this.name + "OpenCloseButton'align=right src= '" + this.image_dir + "close_up_small_blue.gif'>");
   document.writeln("          <b>" + gsSequenceHeading + "</b>");
   document.writeln("          <span id='"+this.name+"jSequenceSummaryText'></span>");
   document.writeln("    </tr>");
   document.writeln("    <tr HELPTAG style='display:none'>");
   document.writeln("       <td>" + gsSequenceHelp + "<td>");
   document.writeln("    </tr>");
   document.writeln("    <tr id='"+this.name+"jSequenceRow' style='display:'>");
   document.writeln("      <td>");
   document.writeln("        <table width=100% border=0 cellspacing=2>");
   document.writeln("           <tr>");
   document.writeln("              <td><input type='radio' id='"+this.name+"jSequenceAutoRadio' name='jSequenceAutoRadio'  onclick='" + this.name + ".onSequenceTypeRadio(0,true)' checked><b>&nbsp;" + gsSequenceAuto + "</b></td>");
   document.writeln("              <td width=70%><div id='"+this.name+"jSequenceTypeList' style='width:100%;' fixheight=16 onselect='" + this.name + ".onSequenceSelect()'> </div> </td>");
   document.writeln("           </tr>");
   document.writeln("           <tr HELPTAG style='display:none'>");
   document.writeln("              <td colspan=2>" + gsSequenceSelectAutoHelp + "<td>");
   document.writeln("           </tr>");
   document.writeln("           <tr>");
   document.writeln("              <td><input type='radio' id='"+this.name+"jSequenceAutoRadio' name='jSequenceAutoRadio' onclick='" + this.name + ".onSequenceTypeRadio(1,true)'><b>" + gsSequenceManual + "</b>&nbsp;</td>");
   document.writeln("              <td valign=bottom>&nbsp;<span id='"+this.name+"jSequenceVectorSelectedSpan'></span></td>");
   document.writeln("           </tr>");
   document.writeln("           <tr>");
   document.writeln("              <td colspan=2 align=center><input type='button' onclick='" + this.name + ".onSequenceSelectVector()' id='"+this.name+"jSequenceSelectVectorButton' value='"+ gsSelectPolyline + "'></td>");
   document.writeln("           </tr>");
   document.writeln("           <tr HELPTAG style='display:none'>");
   document.writeln("              <td colspan=2>" + gsSequenceSelectPolylineHelp + "<td>");
   document.writeln("           </tr>");

   if (this.showLockStartPoints)
      {
      document.writeln("           <tr>");
      document.writeln("              <td colspan=2><input type='checkbox' id='"+this.name+"jSequenceLockStartPointsCheck' name='jSequenceLockStartPointsCheck' onclick='" + this.name + ".onSequenceLockStartPoints()' checked>&nbsp;<b>" + gsLockStartPoints + "</b>&nbsp;</td>");
      document.writeln("           </tr>");
      document.writeln("           <tr HELPTAG style='display:none'>");
      document.writeln("              <td colspan=2>" + gsLockStartPointsHelp + "<td>");
      document.writeln("           </tr>");
      }

   document.writeln("        </table>");
   document.writeln("    </tr>");
   document.writeln("</table>");

   document.writeln("</td>");
   document.writeln("</tr>");
}

// === cutSequence_onCutDirectionTerseClick ===============
//
// User is toggling display of sequencing data
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_onCutDirectionTerseClick()
{
   ToggleVisibility(this.objSequenceRow);

   if (this.objSequenceRow.style.display == "none" )
      {
      this.OpenCloseButton.src = artcam.HtmlRootDir() + "..\\SharedImages\\open_down_small_blue.gif";
      this.OpenCloseButton.border = 0;
      }
   else
      {
      this.OpenCloseButton.src = artcam.HtmlRootDir() + "..\\SharedImages\\close_up_small_blue.gif";
      this.OpenCloseButton.border = 0;
      }


   this.objSequenceTypeList.Refresh();

}

// === cutSequence_onSequenceSelect ============================================
//
// called when user selects a sequence from the drop down list
//
//
// History
// Who When     What
// --- -------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_onSequenceSelect()
{

   this.autoSequenceType = this.objSequenceTypeList.GetSelectedIndex(); 

   // update form this object is in if there is a callback registered
   if (this.doCallback)
      {
      this.updateCallback();
      }

}

// === cutSequence_PopulateSequenceList ====================================
//
// Put all the options into the sequence type selection list
//
//
// History
// Who When     What
// --- -------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_PopulateSequenceList()
{

   this.objSequenceTypeList.Empty();

   this.objSequenceTypeList.AddRow(gsOptionOptimise ,0);
   this.objSequenceTypeList.AddRow(gsOptionPreserveTextOrder,1);
   this.objSequenceTypeList.AddRow(gsOptionLeftRight,2);
   this.objSequenceTypeList.AddRow(gsOptionRightLeft,3);
   this.objSequenceTypeList.AddRow(gsOptionBottomTop,4);
   this.objSequenceTypeList.AddRow(gsOptionTopBottom,5);
   this.objSequenceTypeList.AddRow(gsOptionSpiralOut,6);
   this.objSequenceTypeList.AddRow(gsOptionSpiralIn ,7);

   // this ensure list is redrawn correctly
   this.objSequenceTypeList.SetSelectedIndex(this.autoSequence ? 0 : 1);
}    


// === cutSequence_onSequenceTypeRadio =====================================
//
// called when user changes between auto and manual for sequencing
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_onSequenceTypeRadio(which_radio, do_update)
{
   if (which_radio == 0)
      {
      this.objSequenceSelectVectorButton.disabled = true;
      this.objSequenceAutoRadio[0].checked = true;
      this.objSequenceTypeList.SetDisabled(false);
      this.objSequenceVectorSelectedSpan.innerHTML ="";

      this.autoSequence = true;

      if (this.SequenceDriveVectorId != "")
         this.SetSequenceVectorArrows(this.SequenceDriveVectorId,false);
      }
   else
      {
      this.objSequenceSelectVectorButton.disabled = false;
      this.objSequenceAutoRadio[1].checked = true;
      this.objSequenceTypeList.SetDisabled(true);

      if (this.getManualSequenceVectorId() == "")
         this.objSequenceVectorSelectedSpan.innerHTML = gsNoSequenceVectorSelected;
      else
         {
         this.objSequenceVectorSelectedSpan.innerHTML = gsSequenceVectorSelected;
         this.SetSequenceVectorArrows(this.SequenceDriveVectorId,true);
         }
      this.autoSequence = false;
      }

   // update form this object is in if there is a callback registered
   if (do_update && this.doCallback)
      {
      this.updateCallback();
      }

}

// === cutSequence_onSequenceSelectVector =================================
//
// User is entering or canceling vector selection mode
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_onSequenceSelectVector()
{
   // are we in vector selection mode already ?
   if (this.selectingVectors)
      {
      // yes - cancel and return
      this.onCancelVectorSelection();
      return;
      }

   // if we reach here we are entering vector selection - save current toolpath vectors
   this.ToolpathOriginalVectorSelection = artcam.VectorSelection.Selection;

   // clear selection
   artcam.VectorSelection.Selection = "";

   // and change select button into cancel
   this.objSequenceSelectVectorButton.value = gsCancelSelectPolyline;
   this.selectingVectors = true;

   // if we have a selected vector switch off the arrow direction drawing
   if (this.SequenceDriveVectorId != "")
      {
      this.SetSequenceVectorArrows(this.SequenceDriveVectorId,false);
      }

   // clear record of selected vector
   this.objSequenceVectorSelectedSpan.innerHTML ="";
}


// === cutSequence_onCancelVectorSelection =================================
//
// Cancel vector selection mode and update UI
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_onCancelVectorSelection()
{
   // are we in vector selection mode already ?
   if (this.selectingVectors)
      {
      // yes - we cancel selection mode and retore original toolpath vectors
      artcam.VectorSelection.Selection = this.ToolpathOriginalVectorSelection;
      }
   this.objSequenceSelectVectorButton.value = gsSelectPolyline;
   this.selectingVectors = false;
   artcam.Refresh2dView();
   this.ToolpathOriginalVectorSelection = "";
   return;
}


// === cutSequence_VectorSelectionChanged =======================================
//
// Called when vector selection changes. If we are in the mode for selecting
// a sequence drive curve, check selected vector is suitable ....
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_VectorSelectionChanged()
{
    
   if (!this.selectingVectors)
      return;

   if (this.ToolpathOriginalVectorSelection == "") 
      return;

   if (artcam.VectorSelection.Count() != 1)
      {
      artcam.MessageBox(gsSelectSingleVectorError);
      return;
      }

   //alert("Selected vectors:" + artcam.VectorSelection.Selection)

   // check that the vector selected is valid for use as a sequencing drive curve
   var vector_selected = artcam.VectorSelection.Selection;

   if (!this.ValidateVector(vector_selected,true))
      {
      this.SequenceDriveVectorId = ""; // doesnt exist any more
      artcam.VectorSelection.Selection = "";
      artcam.Refresh2dView();
      return "";
      }
   // if we reach here the vector is OK
   this.SequenceDriveVectorId = vector_selected;

   // if we have a selected vector switch off the arrow direction drawing
   this.SetSequenceVectorArrows(this.SequenceDriveVectorId,true);

   // update form this object is in if there is a callback registered
   if (this.doCallback)
      {
      this.updateCallback();
      }

   // restore original toolpath vectors
   artcam.VectorSelection.Selection = this.ToolpathOriginalVectorSelection;
   artcam.Refresh2dView();

   this.objSequenceSelectVectorButton.value = gsSelectPolyline;
   this.selectingVectors = false;

   this.objSequenceVectorSelectedSpan.innerHTML = gsSequenceVectorSelected;

   this.getManualSequenceVectorId();
}

// === cutSequence_ValidateVector =======================================
//
// Called to check if passed vector id would be valid for use as a 
// sequencing drive curve
//
// returns true if passed vector ok else false
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_ValidateVector
   (
   vector_id,
   display_errors
   )
{
   if (!artcam.ObjectWithIdExists(vector_id))
      return false;

   var obj = artcam.GetVectorObjectWithId(vector_id);
   if (obj == null)
      {
      return false;
      }

   //alert(obj.ObjectClass);
   if (obj.ObjectClass != "CVecContour")
      {
      obj=null;
      if (display_errors)
         artcam.MessageBox(gsSelectSingleVectorError);
      return false;
      }

   return true;

}


// === cutSequence_getAutoSequenceType ======================================
//
// get type of automatic sequencing being used
//
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_getAutoSequenceType()
{
   return this.objSequenceTypeList.GetSelectedIndex();
}

// === cutSequence_setAutoSequenceType ======================================
//
// set type of automatic sequencing being used
//
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_setAutoSequenceType(type)
{
   this.objSequenceTypeList.SetSelectedIndex(type);

   this.autoSequenceType = this.objSequenceTypeList.GetSelectedIndex(); 
}


// === cutSequence_getManualSequenceVectorId ================================
//
// return id of vector to use for manual sequencing
//
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_getManualSequenceVectorId()
{
if (this.SequenceDriveVectorId == "")
   return "";

// check that vector exists and is NOT a group etc
if (!this.ValidateVector(this.SequenceDriveVectorId,true))
   {
   this.SequenceDriveVectorId = "";
   return "";
   }

return this.SequenceDriveVectorId;
}


// === cutSequence_UpdateParametersFromToolpath ===================================
//
// Update sequence parameters from passed toolpath
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
// ----------------------------------------------------------------------

function cutSequence_UpdateParametersFromToolpath(toolpath)
{
   if (toolpath == null)
      return;

   // restore sequencing options
   this.autoSequence = toolpath.GetFlag('AutoSequence',this.autoSequence);

   this.setAutoSequenceType(toolpath.GetInteger('SequenceOptionType', this.getAutoSequenceType()));

   if (this.autoSequence)
      {
      this.SequenceDriveVectorId = "";
      }
   else
      {
      // this was a manually sequenced toolpath - get sequencing vector
      var vector_id = toolpath.GetString('SequenceVectorId',"");
      // check valid but dont display error messages
      if (!this.ValidateVector(vector_id,false))
         {
         // invalid - revert to automatic sequencing
         this.SequenceDriveVectorId = "";
         this.autoSequence = true;  
         }
      else
         {
         // this is valid 
         this.SequenceDriveVectorId = vector_id;
         this.SetSequenceVectorArrows(this.SequenceDriveVectorId,true);
         }
      }


   this.lockStartPoints = toolpath.GetFlag('SequenceLockStartPoints', this.lockStartPoints);


   // now update visibility of options ...
   this.onSequenceTypeRadio(this.autoSequence ? 0 : 1,false);

   if (this.showLockStartPoints)
      this.objSequenceLockStartPoints.checked = this.lockStartPoints;
}

// === cutSequence_SaveParametersToToolpath ===================================
//
// Save sequencing parameters to passed toolpath
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
// ----------------------------------------------------------------------

function cutSequence_SaveParametersToToolpath(toolpath)
{
   if (toolpath == null)
      return;

   // save sequencing options
   toolpath.SetFlag('AutoSequence',this.autoSequence);
   toolpath.SetInteger('SequenceOptionType', this.getAutoSequenceType());
   // save manual vector id
   if (this.autoSequence)
      toolpath.SetString('SequenceVectorId', "");
   else
      toolpath.SetString('SequenceVectorId', this.SequenceDriveVectorId);

   toolpath.SetFlag('SequenceLockStartPoints', this.lockStartPoints);

   // set flag indicating if we sequence against nodes in polyline or any point on span
   // if user chooses manula sequencing just use nodes
   toolpath.SetFlag('SequenceUseVectorNodes', !this.autoSequence);

}

// === cutSequence_onSequenceTypeRadio =====================================
//
// called when user changes toggles locking of start points
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 26/09/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_onSequenceLockStartPoints()
{

 if (this.showLockStartPoints)
    this.lockStartPoints = this.objSequenceLockStartPoints.checked;


 // update form this object is in if there is a callback registered
 if (this.doCallback)
    {
    this.updateCallback();
    }

}


// === cutSequence_SetSequenceVectorArrows =======================================
//
// Switch on / off drawing of arrows to show direction of sequencing vector
//
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 03/12/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_SetSequenceVectorArrows
   (
   vector_id,
   draw_arrows
   )
{
   if (!artcam.ObjectWithIdExists(vector_id))
      return false;

   var obj = artcam.GetVectorObjectWithId(vector_id);
   if (obj == null)
      {
      return false;
      }

   //alert(obj.ObjectClass);
   if (obj.ObjectClass != "CVecContour")
      {
      obj=null;
      return false;
      }

   obj.DirectionArrows = draw_arrows ? 3 : 0;

   artcam.Refresh2dView();

   return true;

}

// === cutSequence_CleanUp =======================================
//
// Switch  off drawing of arrows to show direction of sequencing vector
//
//
// History
// Who When       What
// --- ---------- ---------------------------------------------------------
// bem 03/12/2003 Written
//-----------------------------------------------------------------------------

function cutSequence_CleanUp
   (
   )
{
// if we have a selected vector switch off the arrow direction drawing
if (this.SequenceDriveVectorId != "")
   {
   this.SetSequenceVectorArrows(this.SequenceDriveVectorId,false);
   }
}